import mongoose from 'mongoose';

const MONGODB_URI = process.env.MONGODB_URI;

// Allow build to pass without DB connection (it will fail at runtime if not set)
// This is necessary because Vercel builds the app before you set the env vars in the dashboard
if (!MONGODB_URI && process.env.NODE_ENV === 'production' && process.env.NEXT_PHASE !== 'phase-production-build') {
    // Only throw if we are actually running in production, not during build
} else if (!MONGODB_URI) {
    // If we are in local dev, or building, we might want to skip this error or warn
    // console.warn("MONGODB_URI is not set."); 
}

interface MongooseCache {
  conn: typeof mongoose | null;
  promise: Promise<typeof mongoose> | null;
}

declare global {
  var mongoose: MongooseCache;
}

let cached = global.mongoose;

if (!cached) {
  cached = global.mongoose = { conn: null, promise: null };
}

export async function connectToDatabase() {
  if (cached.conn) {
    return cached.conn;
  }

  if (!cached.promise) {
    const opts = {
      bufferCommands: false,
    };

    cached.promise = mongoose.connect(MONGODB_URI || "", opts).then((mongoose) => {
      return mongoose;
    });
  }
  try {
    cached.conn = await cached.promise;
  } catch (e) {
    cached.promise = null;
    throw e;
  }

  return cached.conn;
}
