'use client';

import Link from 'next/link';
import { useState, useEffect, useRef } from 'react';
import { useToast } from '@/app/context/ToastContext';
import { Send, Clock, CheckCircle, XCircle, AlertCircle, Inbox } from 'lucide-react';

export default function InboxPage() {
  const [requests, setRequests] = useState<any[]>([]);
  const [selectedRequest, setSelectedRequest] = useState<any>(null);
  const [messages, setMessages] = useState<any[]>([]);
  const [input, setInput] = useState('');
  const [loading, setLoading] = useState(true);
  const chatRef = useRef<HTMLDivElement>(null);
  const { addToast } = useToast();

  // Fetch requests list
  useEffect(() => {
    fetch('/api/inbox')
      .then(res => res.json())
      .then(data => {
        setRequests(data.requests || []);
        setLoading(false);
      });
  }, []);

  // Fetch chat messages when a request is selected
  useEffect(() => {
    if (!selectedRequest) return;
    
    const fetchChat = async () => {
        const res = await fetch(`/api/services/chat?requestId=${selectedRequest.id}`);
        const data = await res.json();
        setMessages(data.messages || []);
    };

    fetchChat();
    const interval = setInterval(fetchChat, 3000); // Poll every 3s
    return () => clearInterval(interval);
  }, [selectedRequest]);

  // Scroll to bottom
  useEffect(() => {
    if (chatRef.current) {
        chatRef.current.scrollTop = chatRef.current.scrollHeight;
    }
  }, [messages]);

  const handleSend = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!input.trim() || !selectedRequest) return;

    try {
        const res = await fetch('/api/services/chat', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                requestId: selectedRequest.id,
                message: input
            })
        });
        
        if (res.ok) {
            setInput('');
            // Optimistic update could go here
        }
    } catch (error) {
        addToast("Failed to send", 'error');
    }
  };

  const getStatusColor = (status: string) => {
      switch(status) {
          case 'pending': return 'text-yellow-400';
          case 'completed': return 'text-green-400';
          case 'cancelled': return 'text-red-400';
          case 'in_progress': return 'text-blue-400';
          default: return 'text-gray-400';
      }
  };

  if (loading) return <div className="text-center py-20 text-white">Loading inbox...</div>;

  return (
    <div className="max-w-7xl mx-auto px-4 py-8 h-[calc(100vh-5rem)] flex gap-6">
      {/* Sidebar List */}
      <div className={`w-full md:w-1/3 bg-white/5 border border-white/10 rounded-2xl overflow-hidden flex flex-col ${selectedRequest ? 'hidden md:flex' : 'flex'}`}>
        <div className="p-4 border-b border-white/10 bg-black/20">
            <h2 className="text-xl font-bold text-white">Your Requests</h2>
        </div>
        <div className="flex-1 overflow-y-auto">
            {requests.length === 0 ? (
                <div className="p-8 text-center text-gray-500">No requests yet.</div>
            ) : (
                requests.map(req => (
                    <button
                        key={req.id}
                        onClick={() => setSelectedRequest(req)}
                        className={`w-full text-left p-4 border-b border-white/5 hover:bg-white/5 transition-colors ${selectedRequest?.id === req.id ? 'bg-white/10 border-l-4 border-l-purple-500' : ''}`}
                    >
                        <div className="flex justify-between items-start mb-1">
                            <span className="font-bold text-white truncate">{req.serviceTitle}</span>
                            <span className="text-xs text-gray-500">{new Date(req.updatedAt).toLocaleDateString()}</span>
                        </div>
                        <div className="flex justify-between items-center">
                            <span className="text-xs text-gray-400 truncate max-w-[150px]">{req.id.slice(-6)}</span>
                            <span className={`text-xs capitalize font-medium ${getStatusColor(req.status)}`}>
                                {req.status.replace('_', ' ')}
                            </span>
                        </div>
                    </button>
                ))
            )}
        </div>
      </div>

      {/* Chat Area */}
      <div className={`w-full md:w-2/3 bg-white/5 border border-white/10 rounded-2xl overflow-hidden flex flex-col ${!selectedRequest ? 'hidden md:flex' : 'flex'}`}>
        {!selectedRequest ? (
            <div className="flex-1 flex items-center justify-center text-gray-500 flex-col gap-4">
                <div className="w-16 h-16 rounded-full bg-white/5 flex items-center justify-center">
                    <Inbox size={32} />
                </div>
                Select a request to view details and chat
            </div>
        ) : (
            <>
                {/* Chat Header */}
                <div className="p-4 border-b border-white/10 bg-black/20 flex justify-between items-center">
                    <div className="flex items-center gap-3">
                        <button onClick={() => setSelectedRequest(null)} className="md:hidden text-gray-400 hover:text-white">
                            <ArrowLeftCircle size={24} />
                        </button>
                        <div>
                            <h2 className="font-bold text-white">{selectedRequest.serviceTitle}</h2>
                            <p className="text-xs text-gray-400">ID: {selectedRequest.id}</p>
                        </div>
                    </div>
                    <div className={`px-3 py-1 rounded-full text-xs font-bold border ${
                        selectedRequest.status === 'pending' ? 'border-yellow-500/30 bg-yellow-500/10 text-yellow-400' :
                        selectedRequest.status === 'completed' ? 'border-green-500/30 bg-green-500/10 text-green-400' :
                        'border-gray-500/30 bg-gray-500/10 text-gray-400'
                    }`}>
                        {selectedRequest.status.toUpperCase().replace('_', ' ')}
                    </div>
                </div>

                {/* Messages */}
                <div ref={chatRef} className="flex-1 overflow-y-auto p-6 space-y-4">
                    {/* Initial Details Block */}
                    <div className="bg-white/5 border border-white/10 rounded-xl p-4 mb-6">
                        <h3 className="text-xs font-bold text-gray-500 uppercase mb-2">Original Request</h3>
                        <p className="text-gray-300 text-sm whitespace-pre-wrap">{selectedRequest.initialDetails}</p>
                    </div>

                    {messages.map((msg, i) => {
                        const isSystem = msg.senderRole === 'system';
                        const isMe = msg.senderId === selectedRequest.userId;
                        
                        // Check if this message is a "Payment Request" (detected by content string for now)
                        const isPaymentRequest = msg.message.includes("payment request for $");
                        const priceMatch = msg.message.match(/\$(\d+)/);
                        const price = priceMatch ? priceMatch[1] : null;

                        return (
                            <div key={msg.id} className={`flex flex-col ${isSystem ? 'items-center' : msg.senderRole === 'admin' || msg.senderRole === 'staff' ? 'items-start' : 'items-end'}`}>
                                {isSystem ? (
                                    <div className="text-xs text-gray-500 bg-black/20 px-3 py-1 rounded-full mb-2">
                                        {msg.message}
                                    </div>
                                ) : (
                                    <div className={`max-w-[80%] p-3 rounded-2xl text-sm ${
                                        msg.senderRole === 'admin' || msg.senderRole === 'staff' 
                                            ? 'bg-purple-600/20 border border-purple-500/30 text-purple-100 rounded-tl-none'
                                            : 'bg-blue-600 text-white rounded-tr-none'
                                    }`}>
                                        {msg.senderRole === 'admin' && <div className="text-[10px] font-bold text-purple-300 mb-1">SUPPORT STAFF</div>}
                                        {msg.message}
                                    </div>
                                )}

                                {/* Show PAY button if this is a system payment message AND status is awaiting_payment */}
                                {isPaymentRequest && selectedRequest.status === 'awaiting_payment' && price && (
                                    <div className="mt-2 p-4 bg-green-500/10 border border-green-500/20 rounded-xl flex items-center justify-between gap-4 w-fit">
                                        <div>
                                            <p className="text-xs text-green-400 font-bold uppercase">Payment Required</p>
                                            <p className="text-xl font-bold text-white">${price}</p>
                                        </div>
                                        <Link 
                                            href={`/checkout/request/${selectedRequest.id}`}
                                            className="bg-green-600 hover:bg-green-500 text-white px-4 py-2 rounded-lg font-bold text-sm block text-center"
                                        >
                                            Pay Now
                                        </Link>
                                    </div>
                                )}
                            </div>
                        );
                    })}
                </div>

                {/* Input */}
                <form onSubmit={handleSend} className="p-4 border-t border-white/10 bg-black/20 flex gap-2">
                    <input
                        type="text"
                        value={input}
                        onChange={e => setInput(e.target.value)}
                        placeholder="Type a message..."
                        className="flex-1 bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:outline-none focus:border-purple-500 transition-colors"
                    />
                    <button 
                        type="submit"
                        disabled={!input.trim()}
                        className="bg-purple-600 hover:bg-purple-500 text-white p-3 rounded-xl disabled:opacity-50 transition-colors"
                    >
                        <Send size={20} />
                    </button>
                </form>
            </>
        )}
      </div>
    </div>
  );
}

// Helper icon
const ArrowLeftCircle = ({ size }: { size: number }) => (
    <svg xmlns="http://www.w3.org/2000/svg" width={size} height={size} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><circle cx="12" cy="12" r="10"/><path d="M16 12H8"/><path d="M12 8l-4 4 4 4"/></svg>
);
