'use client';

import Link from 'next/link';
import Image from 'next/image';
import { useRouter, usePathname } from 'next/navigation';
import { useEffect, useState } from 'react';
import { motion } from 'framer-motion';
import { 
  Home, BookOpen, Gift, MessageSquare, History, Sparkles,
  Inbox, User, Shield, LogOut, LogIn, UserPlus, Menu, X 
} from 'lucide-react';

export default function Navbar() {
  const [user, setUser] = useState<any>(null);
  const [scrolled, setScrolled] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [unreadCount, setUnreadCount] = useState(0);
  const router = useRouter();
  const pathname = usePathname();

  useEffect(() => {
    fetch('/api/auth/me')
      .then((res) => res.json())
      .then((data) => setUser(data.user));

    const fetchUnread = () => {
        fetch('/api/inbox/unread')
            .then(res => res.json())
            .then(data => setUnreadCount(data.count || 0))
            .catch(() => {});
    };

    fetchUnread();
    const interval = setInterval(fetchUnread, 10000); // Poll every 10s

    const handleScroll = () => setScrolled(window.scrollY > 20);
    window.addEventListener('scroll', handleScroll);
    return () => {
        window.removeEventListener('scroll', handleScroll);
        clearInterval(interval);
    };
  }, []);

  const handleLogout = async () => {
    await fetch('/api/auth/logout', { method: 'POST' });
    setUser(null);
    router.push('/login');
    router.refresh();
  };

  const navLinks = [
    { href: '/', label: 'Store', icon: Home },
    { href: '/services', label: 'Services', icon: Sparkles },
    { href: '/tutorial', label: 'Tutorial', icon: BookOpen },
    { href: '/free-script', label: 'Free Script', icon: Gift },
    { href: '/community', label: 'Community', icon: MessageSquare },
    { href: '/updates', label: 'Updates', icon: History },
  ];

  return (
    <motion.nav 
      initial={{ y: -100 }}
      animate={{ y: 0 }}
      transition={{ type: 'spring', stiffness: 100, damping: 20 }}
      className={`fixed top-0 w-full z-50 transition-all duration-300 ${
        scrolled 
          ? 'bg-black/80 backdrop-blur-lg border-b border-white/5 shadow-lg shadow-purple-500/5' 
          : 'bg-transparent'
      }`}
    >
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex items-center justify-between h-16">
          {/* Logo */}
          <div className="flex items-center">
            <Link href="/" className="flex items-center gap-2 group">
              <div className="relative w-8 h-8 rounded-lg overflow-hidden shadow-lg group-hover:shadow-purple-500/50 transition-all group-hover:scale-110">
                 <Image 
                    src="/logo.gif" 
                    alt="Logo" 
                    fill 
                    className="object-cover"
                    onError={(e) => {
                        e.currentTarget.style.display = 'none'; 
                    }} 
                 />
              </div>
               
               <div className="w-8 h-8 bg-gradient-to-br from-purple-500 to-pink-500 rounded-lg flex items-center justify-center text-white font-bold text-xl shadow-lg group-hover:shadow-purple-500/50 transition-all group-hover:scale-110 absolute opacity-0">
                J
              </div>

              <span className="text-xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-white to-gray-400 group-hover:to-white transition-all">
                JomHUB
              </span>
            </Link>

            {/* Desktop Nav */}
            <div className="hidden md:flex ml-10 space-x-1">
              {navLinks.map((link) => {
                const isActive = pathname === link.href;
                const Icon = link.icon;
                return (
                  <Link 
                    key={link.href} 
                    href={link.href}
                    className={`relative px-3 py-2 rounded-lg text-sm font-medium transition-all flex items-center gap-1.5 overflow-hidden group ${
                      isActive ? 'text-white' : 'text-gray-400 hover:text-white'
                    }`}
                  >
                    <span className="relative z-10 flex items-center gap-1.5">
                        <Icon size={16} className={isActive ? 'text-purple-400' : 'group-hover:text-purple-400 transition-colors'} />
                        {link.label}
                    </span>
                    {isActive && (
                      <motion.div 
                        layoutId="navbar-active"
                        className="absolute inset-0 bg-white/10 rounded-lg"
                        transition={{ type: 'spring', bounce: 0.2, duration: 0.6 }}
                      />
                    )}
                  </Link>
                );
              })}
            </div>
          </div>

          {/* User Menu (Desktop) */}
          <div className="hidden md:flex items-center space-x-4">
            {user ? (
              <div className="flex items-center gap-3 bg-white/5 border border-white/10 rounded-full pl-4 pr-2 py-1.5">
                <span className="text-xs font-medium text-gray-300 max-w-[100px] truncate">
                    {user.email.split('@')[0]}
                </span>
                
                <div className="h-4 w-[1px] bg-white/10"></div>

                <div className="flex items-center gap-1">
                  <Link href="/inbox" title="Inbox" className="p-1.5 text-gray-400 hover:text-white hover:bg-white/10 rounded-full transition-all relative">
                    <Inbox size={16} />
                    {unreadCount > 0 && (
                        <span className="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full border border-black"></span>
                    )}
                  </Link>
                  <Link href="/profile" title="Profile" className="p-1.5 text-gray-400 hover:text-white hover:bg-white/10 rounded-full transition-all">
                    <User size={16} />
                  </Link>
                  {(user.role === 'admin' || user.role === 'staff') && (
                    <Link href="/admin" title="Admin" className="p-1.5 text-purple-400 hover:text-purple-300 hover:bg-purple-500/10 rounded-full transition-all">
                      <Shield size={16} />
                    </Link>
                  )}
                  <button
                    onClick={handleLogout}
                    className="p-1.5 text-red-400 hover:text-red-300 hover:bg-red-500/10 rounded-full transition-all"
                    title="Logout"
                  >
                    <LogOut size={16} />
                  </button>
                </div>
              </div>
            ) : (
              <div className="flex items-center gap-2">
                <Link href="/login" className="text-sm font-medium text-gray-300 hover:text-white transition-colors flex items-center gap-1">
                  <LogIn size={16} /> Login
                </Link>
                <Link
                  href="/register"
                  className="text-sm font-bold bg-white text-black px-4 py-2 rounded-full hover:bg-gray-200 transition-transform hover:scale-105 active:scale-95 flex items-center gap-1"
                >
                   Register <UserPlus size={16} />
                </Link>
              </div>
            )}
          </div>

          {/* Mobile Menu Button */}
          <div className="md:hidden flex items-center">
            <button 
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              className="p-2 rounded-md text-gray-400 hover:text-white hover:bg-white/10"
            >
              {mobileMenuOpen ? <X /> : <Menu />}
            </button>
          </div>
        </div>
      </div>

      {/* Mobile Menu */}
      {mobileMenuOpen && (
        <motion.div 
          initial={{ opacity: 0, height: 0 }}
          animate={{ opacity: 1, height: 'auto' }}
          exit={{ opacity: 0, height: 0 }}
          className="md:hidden bg-black/95 backdrop-blur-xl border-b border-white/10"
        >
          <div className="px-4 pt-2 pb-4 space-y-1">
            {navLinks.map((link) => (
               <Link 
                 key={link.href}
                 href={link.href}
                 onClick={() => setMobileMenuOpen(false)}
                 className={`block px-3 py-2 rounded-md text-base font-medium flex items-center gap-2 ${
                    pathname === link.href ? 'bg-white/10 text-white' : 'text-gray-300 hover:bg-white/5 hover:text-white'
                 }`}
               >
                 <link.icon size={18} /> {link.label}
               </Link>
            ))}
            <div className="border-t border-white/10 my-2 pt-2">
                {user ? (
                    <>
                        <Link href="/profile" className="block px-3 py-2 text-gray-300 hover:text-white flex items-center gap-2">
                            <User size={18} /> Profile
                        </Link>
                        {(user.role === 'admin' || user.role === 'staff') && (
                            <Link href="/admin" className="block px-3 py-2 text-purple-400 flex items-center gap-2">
                                <Shield size={18} /> Admin
                            </Link>
                        )}
                        <button onClick={handleLogout} className="w-full text-left px-3 py-2 text-red-400 flex items-center gap-2">
                            <LogOut size={18} /> Logout
                        </button>
                    </>
                ) : (
                    <div className="grid grid-cols-2 gap-2 px-2">
                        <Link href="/login" className="text-center py-2 text-gray-300 bg-white/5 rounded-lg">Login</Link>
                        <Link href="/register" className="text-center py-2 text-black bg-white rounded-lg font-bold">Register</Link>
                    </div>
                )}
            </div>
          </div>
        </motion.div>
      )}
    </motion.nav>
  );
}
