'use client';

import { useSearchParams, useRouter } from 'next/navigation';
import { useState, useEffect, Suspense } from 'react';
import Link from 'next/link';
import { CreditCard, Box, Bitcoin, Cat, Link as LinkIcon } from 'lucide-react';
import { PayPalButtons } from "@paypal/react-paypal-js";

function CheckoutContent() {
  const searchParams = useSearchParams();
  const router = useRouter();
  const productSlug = searchParams.get('product');
  
  const [product, setProduct] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [paymentMethod, setPaymentMethod] = useState<'robux' | 'crypto' | 'pets' | 'paypal_link'>('robux');
  const [transactionId, setTransactionId] = useState('');
  const [note, setNote] = useState('');
  const [file, setFile] = useState<File | null>(null);
  const [error, setError] = useState('');

  useEffect(() => {
    if (productSlug) {
        fetch('/api/products')
            .then(res => res.json())
            .then(data => {
                const found = data.products.find((p: any) => p.slug === productSlug);
                setProduct(found || null);
                setLoading(false);
            })
            .catch(() => setLoading(false));
    } else {
        setLoading(false);
    }
  }, [productSlug]);

  const handleApprove = async (orderId: string) => {
    if (!product) return;
    setLoading(true);
    const formData = new FormData();
    formData.append('productSlug', product.slug);
    formData.append('paymentMethod', 'paypal_smart');
    formData.append('transactionId', orderId);
    formData.append('note', 'Paid via PayPal Smart Buttons');
    
    const res = await fetch('/api/submissions/create', {
      method: 'POST',
      body: formData,
    });

    if (!res.ok) {
      setError('Payment successful but submission failed. Please contact support with ID: ' + orderId);
      setLoading(false);
      return;
    }

    router.push('/inbox');
  };

  if (loading) {
      return <div className="p-10 text-center">Loading product details...</div>;
  }

  if (!productSlug || !product) {
      return (
        <div className="p-10 text-center">
          <p className="text-red-400 mb-4">Product not found or invalid.</p>
          <Link href="/" className="text-white underline">Return Home</Link>
        </div>
      );
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError('');

    const formData = new FormData();
    formData.append('productSlug', product.slug);
    formData.append('paymentMethod', paymentMethod);
    formData.append('transactionId', transactionId);
    formData.append('note', note);
    if (file) {
      formData.append('receipt', file);
    }

    const res = await fetch('/api/submissions/create', {
      method: 'POST',
      body: formData,
    });

    if (!res.ok) {
      const data = await res.json();
      setError(data.error || 'Submission failed');
      setLoading(false);
      return;
    }

    // Redirect to inbox
    router.push('/inbox');
  };

  return (
    <div className="max-w-2xl mx-auto px-4 py-12">
      <div className="bg-white/5 border border-white/10 rounded-2xl p-8">
        <h1 className="text-2xl font-bold mb-2">Manual Checkout</h1>
        <div className="text-sm text-gray-400 mb-6">
          Please follow the instructions carefully.
        </div>

        <div className="bg-blue-500/10 border border-blue-500/20 rounded-xl p-4 mb-8">
          <h3 className="font-semibold text-blue-300 mb-2">Order Summary</h3>
          <div className="flex justify-between text-sm mb-1">
            <span>Product:</span>
            <span className="text-white">{product.name}</span>
          </div>
          <div className="flex justify-between text-sm font-bold">
            <span>Total to Pay:</span>
            <span className="text-white">${product.price}</span>
          </div>
        </div>

        <div className="mb-8 space-y-6">
          <div>
            <h3 className="font-semibold mb-4">Pay Securely with PayPal</h3>
             <div className="max-w-md mx-auto relative z-0">
                <PayPalButtons 
                  style={{ layout: "vertical", color: "gold", shape: "rect", label: "pay" }}
                  createOrder={(data, actions) => {
                    return actions.order.create({
                      intent: "CAPTURE",
                      purchase_units: [
                        {
                          amount: {
                            currency_code: "USD",
                            value: product.price.toString(),
                          },
                          description: product.name,
                        },
                      ],
                    });
                  }}
                  onApprove={async (data, actions) => {
                    if (actions.order) {
                      const details = await actions.order.capture();
                      await handleApprove(details.id || data.orderID);
                    }
                  }}
                  onError={(err) => {
                    console.error(err);
                    setError("PayPal Error: Could not process payment.");
                  }}
                />
             </div>
          </div>

          <div className="relative">
             <div className="absolute inset-0 flex items-center">
               <div className="w-full border-t border-white/10"></div>
             </div>
             <div className="relative flex justify-center text-sm">
               <span className="px-2 bg-black text-gray-500">OR Manual Payment</span>
             </div>
          </div>

          <div>
            <h3 className="font-semibold mb-2">Manual Payment Instructions</h3>
            
            <div className="flex gap-2 mb-6 bg-black/20 p-1 rounded-lg">
              <button
                onClick={() => setPaymentMethod('robux')}
                className={`flex-1 flex items-center justify-center gap-2 py-2 rounded-md text-sm font-medium transition-colors ${
                  paymentMethod === 'robux' ? 'bg-white/10 text-white' : 'text-gray-400 hover:text-gray-200'
                }`}
              >
                <Box size={16} /> Robux
              </button>
              <button
                onClick={() => setPaymentMethod('crypto')}
                className={`flex-1 flex items-center justify-center gap-2 py-2 rounded-md text-sm font-medium transition-colors ${
                  paymentMethod === 'crypto' ? 'bg-white/10 text-white' : 'text-gray-400 hover:text-gray-200'
                }`}
              >
                <Bitcoin size={16} /> Crypto
              </button>
              <button
                onClick={() => setPaymentMethod('pets')}
                className={`flex-1 flex items-center justify-center gap-2 py-2 rounded-md text-sm font-medium transition-colors ${
                  paymentMethod === 'pets' ? 'bg-white/10 text-white' : 'text-gray-400 hover:text-gray-200'
                }`}
              >
                <Cat size={16} /> Pets
              </button>
              <button
                onClick={() => setPaymentMethod('paypal_link')}
                className={`flex-1 flex items-center justify-center gap-2 py-2 rounded-md text-sm font-medium transition-colors ${
                  paymentMethod === 'paypal_link' ? 'bg-white/10 text-white' : 'text-gray-400 hover:text-gray-200'
                }`}
              >
                <LinkIcon size={16} /> Link
              </button>
            </div>

            {paymentMethod === 'robux' && (
              <ol className="list-decimal list-inside text-gray-400 space-y-2 text-sm">
                <li>
                  Purchase the gamepass here: <a href="https://www.roblox.com/game-pass/802447291/Premium-Key-Jomhub" target="_blank" className="text-blue-400 hover:underline">Premium Key Gamepass</a>
                </li>
                <li>Enter your <strong>Roblox Username</strong> below.</li>
                <li>Take a screenshot of the purchase (Required).</li>
                <li>Submit the form for approval.</li>
              </ol>
            )}

            {paymentMethod === 'crypto' && (
               <div className="text-center py-4 text-gray-400 text-sm">
                 Crypto payments coming soon. Please use PayPal or Robux.
               </div>
            )}

             {paymentMethod === 'pets' && (
               <div className="text-center py-4 text-gray-400 text-sm">
                 Pet payments coming soon. Please use PayPal or Robux.
               </div>
            )}

            {paymentMethod === 'paypal_link' && (
              <ol className="list-decimal list-inside text-gray-400 space-y-2 text-sm">
                <li>
                  Click this link to pay via PayPal (Guest supported): <a href="https://www.paypal.com/ncp/payment/BJ6A9MUPLKAUY" target="_blank" className="text-blue-400 hover:underline">PayPal Secure Payment</a>
                </li>
                <li>Complete the payment on the PayPal page.</li>
                <li>Copy the <strong>Transaction ID</strong> from your receipt.</li>
                <li>Paste the ID below and upload a screenshot.</li>
              </ol>
            )}
          </div>
        </div>

        {error && (
            <div className="bg-red-500/10 border border-red-500/20 text-red-400 p-3 rounded-lg mb-6 text-sm">
              {error}
            </div>
        )}

        <form onSubmit={handleSubmit} className="space-y-6">
          {paymentMethod === 'robux' ? (
             <div>
               <label className="block text-sm font-medium text-gray-400 mb-1">Roblox Username <span className="text-red-400">*</span></label>
               <input
                 type="text"
                 value={transactionId}
                 onChange={(e) => setTransactionId(e.target.value)}
                 className="w-full bg-black/20 border border-white/10 rounded-lg px-4 py-2 focus:outline-none focus:border-white/30 transition-colors"
                 placeholder="e.g. RobloxPlayer123"
                 required
               />
             </div>
          ) : (
            <div>
               <label className="block text-sm font-medium text-gray-400 mb-1">Transaction ID / Details <span className="text-red-400">*</span></label>
               <input
                 type="text"
                 value={transactionId}
                 onChange={(e) => setTransactionId(e.target.value)}
                 className="w-full bg-black/20 border border-white/10 rounded-lg px-4 py-2 focus:outline-none focus:border-white/30 transition-colors"
                 required={false}
                 disabled={paymentMethod === 'crypto' || paymentMethod === 'pets'}
               />
            </div>
          )}

          <div>
            <label className="block text-sm font-medium text-gray-400 mb-1">Proof of Purchase (Screenshot) <span className="text-red-400">*</span></label>
            <input
              type="file"
              accept="image/*"
              onChange={(e) => setFile(e.target.files?.[0] || null)}
              className="w-full bg-black/20 border border-white/10 rounded-lg px-4 py-2 focus:outline-none focus:border-white/30 transition-colors text-sm"
              required
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-400 mb-1">Note (Optional)</label>
            <textarea
              value={note}
              onChange={(e) => setNote(e.target.value)}
              className="w-full bg-black/20 border border-white/10 rounded-lg px-4 py-2 focus:outline-none focus:border-white/30 transition-colors h-24 resize-none"
              placeholder="Any additional details..."
            />
          </div>

          <button
            type="submit"
            disabled={loading}
            className="w-full bg-white text-black font-bold py-3 rounded-xl hover:bg-gray-200 transition-colors disabled:opacity-50"
          >
            {loading ? 'Submitting...' : 'Confirm Payment'}
          </button>
        </form>
      </div>
    </div>
  );
}

export default function CheckoutPage() {
  return (
    <Suspense fallback={<div className="p-10 text-center">Loading checkout...</div>}>
      <CheckoutContent />
    </Suspense>
  );
}
