import { NextResponse } from 'next/server';
import { writeFile } from 'fs/promises';
import path from 'path';
import { getSession } from '@/lib/auth';
import { connectToDatabase } from '@/lib/db';
import { Submission } from '@/lib/models';
import { getProduct } from '@/lib/products';
import { rateLimit } from '@/lib/ratelimit';
import { z } from 'zod';

const submissionSchema = z.object({
  productSlug: z.string().min(1),
  transactionId: z.string().min(1),
  note: z.string().optional(),
  paymentMethod: z.string().default('manual'),
});

export async function POST(request: Request) {
  try {
    const session = await getSession();
    if (!session) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    // Rate Limiting: 5 submissions per hour (prevent spamming orders)
    const ip = request.headers.get('x-forwarded-for') || 'unknown';
    if (!rateLimit(ip, 'submission', 5, 60 * 60 * 1000)) {
      return NextResponse.json({ error: "Too many submissions. Please wait a while." }, { status: 429 });
    }

    const formData = await request.formData();
    
    // Manual Validation for FormData (Zod works best with JSON usually, but we can map it)
    const rawData = {
      productSlug: formData.get('productSlug'),
      transactionId: formData.get('transactionId'),
      note: formData.get('note') || undefined,
      paymentMethod: formData.get('paymentMethod') || 'manual',
    };

    const result = submissionSchema.safeParse(rawData);
    if (!result.success) {
       return NextResponse.json({ error: "Invalid fields" }, { status: 400 });
    }

    const { productSlug, transactionId, note, paymentMethod } = result.data;
    const file = formData.get('receipt') as File | null;

    // Note: Logic continues as before...
    const product = getProduct(productSlug);
    if (!product) {
      return NextResponse.json({ error: "Invalid product" }, { status: 400 });
    }

    let receiptFilename = null;
    if (file && file.size > 0) {
      // Check for valid image types
      if (!file.type.startsWith('image/')) {
         return NextResponse.json({ error: "Invalid file type. Only images allowed." }, { status: 400 });
      }
      // Limit size (e.g. 5MB)
      if (file.size > 5 * 1024 * 1024) {
        return NextResponse.json({ error: "File too large (max 5MB)." }, { status: 400 });
      }

      const buffer = Buffer.from(await file.arrayBuffer());
      // Sanitize and unique filename
      const ext = path.extname(file.name) || '.jpg';
      const timestamp = Date.now();
      const safeName = `${session.user.id}_${timestamp}${ext}`;
      receiptFilename = safeName;
      
      await writeFile(
        path.join(process.cwd(), 'private_uploads/receipts', safeName),
        buffer
      );
    }

    await connectToDatabase();
    
    const newSubmission = await Submission.create({
      userId: session.user.id,
      userEmail: session.user.email,
      productId: product.id,
      productName: product.name,
      productSlug: product.slug,
      amount: product.price,
      paymentMethod,
      transactionId,
      note: note || '',
      receiptFilename,
      status: 'pending',
    });

    return NextResponse.json({ success: true, submissionId: newSubmission._id.toString() });
  } catch (error) {
    console.error("Submission error:", error);
    return NextResponse.json({ error: "Internal server error" }, { status: 500 });
  }
}
