import { NextResponse } from 'next/server';
import { getSession } from '@/lib/auth';
import { connectToDatabase } from '@/lib/db';
import { ServiceChat, ServiceRequest } from '@/lib/models';

export async function GET(request: Request) {
  const { searchParams } = new URL(request.url);
  const requestId = searchParams.get('requestId');

  const session = await getSession();
  if (!session) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
  }

  await connectToDatabase();
  const req = await ServiceRequest.findById(requestId);
  
  if (!req) return NextResponse.json({ error: "Not found" }, { status: 404 });

  // Verify access
  if (req.userId !== session.user.id && session.user.role !== 'admin' && session.user.role !== 'staff') {
      return NextResponse.json({ error: "Forbidden" }, { status: 403 });
  }

  const messages = await ServiceChat.find({ requestId }).sort({ createdAt: 1 }).lean();

  // Mark messages as read if viewed by recipient
  if (session.user.role === 'admin' || session.user.role === 'staff') {
      // Admin viewing: Mark user messages as read
      await ServiceChat.updateMany({ requestId, senderRole: 'user', read: false }, { read: true });
  } else {
      // User viewing: Mark admin/system messages as read
      await ServiceChat.updateMany({ requestId, senderRole: { $ne: 'user' }, read: false }, { read: true });
  }

  return NextResponse.json({ 
      messages: messages.map((m: any) => ({
          ...m,
          id: m._id.toString(),
          _id: undefined
      }))
  });
}

export async function POST(request: Request) {
  const session = await getSession();
  if (!session) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
  }

  const { requestId, message } = await request.json();
  await connectToDatabase();

  const req = await ServiceRequest.findById(requestId);
  if (!req) return NextResponse.json({ error: "Not found" }, { status: 404 });

  // Verify access
  if (req.userId !== session.user.id && session.user.role !== 'admin' && session.user.role !== 'staff') {
      return NextResponse.json({ error: "Forbidden" }, { status: 403 });
  }

  const newMsg = await ServiceChat.create({
      requestId,
      senderId: session.user.id,
      senderRole: session.user.role,
      message
  });

  // Update request timestamp
  req.updatedAt = new Date();
  await req.save();

  return NextResponse.json({ success: true, message: newMsg });
}
