import { NextResponse } from 'next/server';
import bcrypt from 'bcryptjs';
import { connectToDatabase } from '@/lib/db';
import { User } from '@/lib/models';
import { login } from '@/lib/auth';
import { rateLimit } from '@/lib/ratelimit';
import { z } from 'zod';

const loginSchema = z.object({
  email: z.string().email(),
  password: z.string().min(1)
});

export async function POST(request: Request) {
  try {
    // Rate Limiting
    const ip = request.headers.get('x-forwarded-for') || 'unknown';
    if (!rateLimit(ip, 'login', 5, 15 * 60 * 1000)) {
      return NextResponse.json({ error: "Too many login attempts. Please try again later." }, { status: 429 });
    }

    const body = await request.json();
    
    // Input Validation
    const result = loginSchema.safeParse(body);
    if (!result.success) {
        return NextResponse.json({ error: "Invalid input" }, { status: 400 });
    }
    const { email, password } = result.data;

    await connectToDatabase();
    const user = await User.findOne({ email });

    if (!user) {
      return NextResponse.json({ error: "Invalid credentials" }, { status: 401 });
    }

    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch) {
      console.log("Password mismatch for user:", email);
      return NextResponse.json({ error: "Invalid credentials" }, { status: 401 });
    }

    if (user.isBanned) {
      return NextResponse.json({ error: "Account banned. Contact support." }, { status: 403 });
    }

    if (user.isDisabled) {
      return NextResponse.json({ error: "Account disabled." }, { status: 403 });
    }

    // Update last login
    user.lastLogin = new Date();
    await user.save();

    // Map _id to id
    const userObject = user.toObject();
    userObject.id = userObject._id.toString();
    delete userObject._id;
    delete userObject.password;
    delete userObject.__v;

    await login(userObject);

    return NextResponse.json({ success: true, user: userObject });
  } catch (error) {
    console.error("Login error:", error);
    return NextResponse.json({ error: "Internal server error" }, { status: 500 });
  }
}
