'use client';

import { useEffect, useState } from 'react';
import { useToast } from '@/app/context/ToastContext';

export default function ManageUpdatesPage() {
  const [updates, setUpdates] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [formData, setFormData] = useState({ version: '', date: '', changes: '' });
  const { addToast } = useToast();

  const fetchUpdates = () => {
    fetch('/api/updates')
      .then((res) => res.json())
      .then((data) => {
        setUpdates(data.updates || []);
        setLoading(false);
      })
      .catch(() => addToast("Failed to fetch updates", 'error'));
  };

  useEffect(() => {
    fetchUpdates();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const changesArray = formData.changes.split('\n').filter(line => line.trim() !== '');
    await fetch('/api/updates', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ ...formData, changes: changesArray })
    });
    setFormData({ version: '', date: '', changes: '' });
    fetchUpdates();
    addToast("Update log added successfully", 'success');
  };

  const handleDelete = async (id: string) => {
    if(!confirm("Delete this update log?")) return;
    await fetch(`/api/updates/${id}`, { method: 'DELETE' });
    fetchUpdates();
    addToast("Update log deleted", 'info');
  };

  if (loading) return <div>Loading...</div>;

  return (
    <div className="max-w-6xl mx-auto px-4 py-12">
      <h1 className="text-3xl font-bold mb-8">Manage Update Logs</h1>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <div className="bg-white/5 border border-white/10 rounded-xl p-6 h-fit">
           <h2 className="text-xl font-bold mb-4">Add New Update</h2>
           <form onSubmit={handleSubmit} className="space-y-4">
             <input 
                type="text" placeholder="Version (e.g. v1.0.0)" className="w-full bg-black/20 border border-white/10 rounded p-2"
                value={formData.version} onChange={e => setFormData({...formData, version: e.target.value})} required
             />
             <input 
                type="date" className="w-full bg-black/20 border border-white/10 rounded p-2 text-white"
                value={formData.date} onChange={e => setFormData({...formData, date: e.target.value})} required
             />
             <textarea 
                placeholder="Changes (One per line)" className="w-full bg-black/20 border border-white/10 rounded p-2 h-32"
                value={formData.changes} onChange={e => setFormData({...formData, changes: e.target.value})} required
             />
             <button type="submit" className="w-full bg-white text-black p-2 rounded font-bold">Add Update</button>
           </form>
        </div>

        <div className="lg:col-span-2 space-y-4">
           {updates.map((u) => (
             <div key={u.id} className="bg-white/5 border border-white/10 rounded-xl p-6">
                <div className="flex justify-between items-start mb-2">
                   <div>
                       <h3 className="font-bold text-lg">{u.version}</h3>
                       <span className="text-sm text-gray-400">{u.date}</span>
                   </div>
                   <button onClick={() => handleDelete(u.id)} className="bg-red-500/20 text-red-400 px-3 py-1 rounded hover:bg-red-500/30">
                      Delete
                   </button>
                </div>
                <ul className="list-disc list-inside text-gray-300 text-sm">
                  {u.changes.map((c: string, i: number) => <li key={i}>{c}</li>)}
                </ul>
             </div>
           ))}
        </div>
      </div>
    </div>
  );
}
